import { useState, useMemo, useEffect } from 'react';
import Table from '@/components/ui/Table';
import Pagination from '@/components/ui/Pagination';
import Select from '@/components/ui/Select';
import Input from '@/components/ui/Input';
import {
    useReactTable,
    getCoreRowModel,
    getFilteredRowModel,
    getPaginationRowModel,
    flexRender,
    ColumnDef,
    FilterFn,
} from '@tanstack/react-table';
import { Avatar, Badge, Button, Drawer } from '@/components/ui';
import { AiFillEye, AiFillDelete, AiFillEdit } from "react-icons/ai";
import { rankItem } from '@tanstack/match-sorter-utils';

// for search filter
import type { InputHTMLAttributes } from 'react';
import { useAppSelector } from '@/store';
import { useNavigate } from 'react-router-dom';
import useCustomToastify from '@/utils/customHook/useCustomToastify';

interface DebouncedInputProps
    extends Omit<InputHTMLAttributes<HTMLInputElement>, 'onChange' | 'size' | 'prefix'> {
    value: string | number;
    onChange: (value: string | number) => void;
    debounce?: number;
}

function DebouncedInput({
    value: initialValue,
    onChange,
    debounce = 500,
    ...props
}: DebouncedInputProps) {
    const [value, setValue] = useState(initialValue);

    useEffect(() => {
        setValue(initialValue);
    }, [initialValue]);

    useEffect(() => {
        const timeout = setTimeout(() => {
            onChange(value);
        }, debounce);

        return () => clearTimeout(timeout);
    }, [value, debounce, onChange]);

    return (
        // <div className="flex justify-end">
        <div className="flex items-center mb-4">
            <Input
                {...props}
                value={value}
                onChange={(e) => setValue(e.target.value)}
            />
        </div>
        // </div>
    );
}

const fuzzyFilter: FilterFn<Person> = (row, columnId, value, addMeta) => {
    const itemValue = row.getValue(columnId) as string;
    const itemRank = rankItem(itemValue, value as string);
    addMeta({ itemRank });
    return itemRank.passed;
};

type Person = {
    user_id: number;
    full_name: string;
    picture: string;
    email: string;
    mobile: number;
    user_role: number;
    status: number;
};
type UserDeatils = {
    user_id: string,
    full_name: string,
    picture: string,
    country: string,
    state: string,
    city: string,
    // zip_code: number,
    street_1: string,
    street_2: string,
    email: string,
    mobile: number,
    status: number,
    user_role: number,
    countryName: string,
    stateName: string,
    cityName: string,
}

const signleUserInitailState={
    user_id: '',
    full_name: '',
    picture: '',
    country: '',
    countryName: '',
    state: '',
    city: '',
    street_1: '',
    street_2: '',
    email: '',
    mobile: 0,
    status: 0,
    user_role: 0,
    cityName: '',
    stateName: '',
}

interface TitleProps {
    userDeatil: UserDeatils;
}

type Option = {
    value: number;
    label: string;
};

const { Tr, Th, Td, THead, TBody } = Table;

const pageSizeOption = [
    { value: 10, label: '10 / page' },
    { value: 20, label: '20 / page' },
    { value: 30, label: '30 / page' },
    { value: 40, label: '40 / page' },
    { value: 50, label: '50 / page' },
];

const UsersTable = () => {
    const customToast = useCustomToastify()
    const theme = useAppSelector((state) => state.theme)
    const navigate = useNavigate()
    const [data, setData] = useState<Person[]>([]);
    const [userDeatil, setUserDeatil] = useState<UserDeatils>({...signleUserInitailState})
    const [globalFilter, setGlobalFilter] = useState('');
    const [pageSize, setPageSize] = useState(10);
    const [totalFiltered, setTotalFiltered] = useState(0)
    const [currentPage, setCurrentPage] = useState(1)

    // Drawer State
    const [drawerIsOpen, setDrawerIsOpen] = useState(false)

    useEffect(() => {
        const timer = setTimeout(() => {
            getUserList()
        }, 500)
        return () => clearTimeout(timer)
    }, [currentPage, pageSize, globalFilter])

    const getUserList = async () => {
        try {
            const api = await fetch(`${import.meta.env.VITE_BASE_API_URL}/users/get-all-users?per_page=${pageSize}&current_page=${currentPage}&search_text=${globalFilter}`, {
                method: "GET",
            });
            const response = await api.json();
            if (!response.error) {
                setData(response.data.users)
                setTotalFiltered(response.data.total)
            } else {
                customToast.danger(response.message)
            }
        } catch (error) {
            console.error(error)
        }
    }
    const getSingleUserDeatils = async (id: string) => {
        try {
            const api = await fetch(`${import.meta.env.VITE_BASE_API_URL}/users/get-single-user?user_id=${id}`, {
                method: "GET",
            });
            const response = await api.json();
            if (!response.error) {
                setUserDeatil(response.data)
                setDrawerIsOpen(true)
            } else {
                customToast.danger(response.message)
            }
        } catch (error) {
            console.error(error)
        }
    }



    const columns = useMemo<ColumnDef<Person>[]>(() => [
        {
            header: 'Sr No',
            accessorKey: 'srno',
            cell: ({ row }) => (table.getState().pagination.pageSize * table.getState().pagination.pageIndex) + table.getRowModel().rows.indexOf(row) + 1,
        },
        {
            header: 'Picture', accessorKey: 'picture',
            cell: ({ row }) => <img src={`${row.original.picture}`} width={80} alt="image" />
        },
        {
            header: 'Name', accessorKey: 'name',
            cell: ({ row }) => row.original.full_name
        },
        { header: 'Email', accessorKey: 'email' },
        { header: 'Mobile', accessorKey: 'mobile' },
        // { header: 'Sex', accessorKey: 'sex' },
        {
            header: 'User Role', accessorKey: 'userrole',
            cell: ({ row }) => import.meta.env.VITE_USER_ROLL.split(",").find((item: string, index: number) => index == row.original.user_role)
        },
        {
            header: 'Status', accessorKey: 'status',
            cell: ({ row }) => row.original.status == 1 ? "Active" : "Inactive"
        },
        {
            header: 'Action',
            accessorKey: 'action',
            cell: ({ row }) => (
                <div className="flex gap-2">
                    <Button
                        size='xs'
                        style={{ backgroundColor: '#e6e1e1', color: 'black' }}
                        icon={<AiFillEye />}
                        onClick={() => {

                            getSingleUserDeatils(row.original.user_id + "")
                        }}
                    ></Button>
                    <Button
                        size='xs'
                        style={{ backgroundColor: '#0d8504', color: 'wheat' }}
                        icon={<AiFillEdit />}
                        onClick={() => { navigate(`/auth/users/updateUsers/${row.original.user_id}`) }}
                    ></Button>
                    <Button size='xs' style={{ backgroundColor: '#bd0b0b', color: 'wheat' }} icon={<AiFillDelete />} ></Button>
                </div>
            ),
        },
    ], []);

    const table = useReactTable({
        data,
        columns,
        state: {
            globalFilter,
        },
        onGlobalFilterChange: setGlobalFilter,
        globalFilterFn: fuzzyFilter,
        getCoreRowModel: getCoreRowModel(),
        getFilteredRowModel: getFilteredRowModel(),
        getPaginationRowModel: getPaginationRowModel(),
        enableColumnResizing: true,
        columnResizeMode: 'onChange',
    });

    const filteredData = table.getFilteredRowModel().rows.map(row => row.original);
    // const totalFiltered = filteredData.length;

    useEffect(() => {
        if (table.getState().pagination.pageIndex >= Math.ceil(totalFiltered / pageSize)) {
            table.setPageIndex(Math.max(Math.ceil(totalFiltered / pageSize) - 1, 0));
        }
    }, [globalFilter, pageSize, totalFiltered, table]);

    const onPaginationChange = (page: number) => {
        console.log("tabele sixe ::::", page)
        setCurrentPage(page)
        // table.setPageIndex(page - 1);
    };

    const onSelectChange = (value = 0) => {
        setPageSize(Number(value));
        table.setPageSize(Number(value));
    };

    const Footer =({userDeatil}:TitleProps)=>{
        return  <div className="flex w-full items-start">
            <Button
                block
                className="mx-2"
                variant="solid"
                onClick={() => {
                    setDrawerIsOpen(false)
                    navigate(`/auth/users/updateUsers/${userDeatil.user_id}`)
                }}
            >
                Edit
            </Button>
            <Button block className="mx-2" onClick={() => {
                setDrawerIsOpen(false)
            }}>
                Cancel
            </Button>

        </div>
    }

    const Title = ({ userDeatil }: TitleProps) => {
        return <div className='text-center'>
            {/* <h4 className="mb-2 text-center">
           

        </h4> */}
            <div className='flex items-center justify-center'>
                <Badge
                    className="mr-4 rounded-full"
                    innerClass={userDeatil.status == 1 ? "bg-emerald-500" : ""}
                    badgeStyle={{ top: '15px', right: '33px', width: '18px', height: '18px' }}
                >
                    <Avatar shape="circle" size={120} className="mr-4"
                        src={`${userDeatil.picture}`}
                    // src="/img/avatars/thumb-1.jpg"
                    />
                </Badge>
            </div>
            <p className='text-2xl my-2'>{userDeatil.full_name}</p>
            <p>{userDeatil.email} | {userDeatil.mobile}</p>
            <p>{import.meta.env.VITE_USER_ROLL.split(",").find((item: string, index: number) => index == userDeatil.user_role)}</p>
        </div>
    }

    return (
        <>
            <div className={`${theme.mode == "dark" ? 'bg-[rgb(31,41,55)]' : 'bg-[rgb(255,255,255)]'} p-3 rounded-2xl`} >
                <div className='mt-3 mb-10 ml-4 flex items-center justify-between'>
                    <h3 className=''><b>All Users</b></h3>
                    <Button className="mr-2 mb-2" size="sm" variant="solid" color={theme ? `${theme.themeColor}-${theme.primaryColorLevel}` : "blue-600"} onClick={() => { navigate('/auth/users/addUsers') }}>
                        Add User
                    </Button>
                </div>
                <div className="flex justify-between">
                    <div style={totalFiltered > 0 ? { minWidth: 150 } : { minWidth: 150, visibility: "hidden" }} >
                        <Select<Option>
                            size="sm"
                            isSearchable={false}
                            value={pageSizeOption.find(
                                (option) =>
                                    option.value ===
                                    table.getState().pagination.pageSize,
                            )}
                            options={pageSizeOption}
                            onChange={(option) => onSelectChange(option?.value)}
                        />
                    </div>
                    <DebouncedInput
                        value={globalFilter ?? ''}
                        className="p-2 font-lg shadow border border-block w-72"
                        placeholder="Search all columns..."
                        onChange={(value) => {
                            console.log("option", value)
                            setGlobalFilter(String(value))
                        }}
                    />
                </div>
                <Table>
                    <THead>
                        {table.getHeaderGroups().map((headerGroup) => (
                            <Tr key={headerGroup.id}>
                                {headerGroup.headers.map((header) => (
                                    <Th key={header.id} colSpan={header.colSpan} style={{ position: 'relative', width: header.getSize(), }}>
                                        {flexRender(header.column.columnDef.header, header.getContext())}
                                        {header.column.getCanResize() && (
                                            <div
                                                className={`table-resizer cursor-all-scroll ${header.column.getIsResizing() ? 'isResizing' : ''}`}
                                                onMouseDown={header.getResizeHandler()}
                                                onTouchStart={header.getResizeHandler()}
                                            ></div>
                                        )}
                                    </Th>
                                ))}
                            </Tr>
                        ))}
                    </THead>
                    <TBody>
                        {table.getRowModel().rows.map((row) => (
                            <Tr key={row.id}>
                                {row.getVisibleCells().map((cell) => (
                                    <Td key={cell.id} style={{ width: cell.column.getSize() }}>
                                        {flexRender(cell.column.columnDef.cell, cell.getContext())}
                                    </Td>
                                ))}
                            </Tr>
                        ))}
                    </TBody>
                </Table>
                {totalFiltered > 0 && (

                    <div className="flex items-center justify-between mt-4 mb-16">
                        <div>
                            Showing 1 to {table.getState().pagination.pageSize} of {totalFiltered} entries
                        </div>
                        <Pagination
                            pageSize={pageSize}
                            currentPage={table.getState().pagination.pageIndex + 1}
                            total={totalFiltered}
                            onChange={onPaginationChange}
                        />
                    </div>
                )}
            </div>
            <Drawer
                title={<Title
                    userDeatil={userDeatil}
                />}
                isOpen={drawerIsOpen}
                footer={<Footer
                    userDeatil={userDeatil}
                />}
                closable={false}
                headerClass="!items-center !justify-center !bg-gray-100 dark:!bg-gray-700"
                footerClass="!border-t-0 !p-3"
                onClose={() => {
                    setDrawerIsOpen(false);
                    setUserDeatil({...signleUserInitailState})
                }}
                onRequestClose={() => {
                    setDrawerIsOpen(false);
                    setUserDeatil({...signleUserInitailState})
                }}
            >
                <div>
                    <h2 className="text-2xl font-bold mb-4">Address</h2>
                    <div>
                        <p className='text-base'>{userDeatil.street_1}{userDeatil.street_2 ? `, ${userDeatil.street_2}` : null}</p>
                        <p className='text-base my-1'><span className='font-semibold text-base w-20 inline-block'>Country :</span> {userDeatil.countryName}</p>
                        <p className='text-base my-1'><span className='font-semibold text-base w-20 inline-block'>State :</span> {userDeatil.stateName}</p>
                        <p className='text-base my-1'><span className='font-semibold text-base w-20 inline-block'>City :</span> {userDeatil.cityName}</p>
                        {/* <p className='text-base my-1'><span className='font-semibold text-base w-20 inline-block'>Zip code :</span> 416003</p> */}
                    </div>
                </div>
            </Drawer>
        </>
    );
};

export default UsersTable;
