
import { useMemo, useState, useEffect } from 'react'
import Table from '@/components/ui/Table'
import Input from '@/components/ui/Input'
import Pagination from '@/components/ui/Pagination';
import Select from '@/components/ui/Select';
import Dialog from '@/components/ui/Dialog'
import {
    useReactTable,
    getCoreRowModel,
    getFilteredRowModel,
    getFacetedRowModel,
    getFacetedUniqueValues,
    getFacetedMinMaxValues,
    getPaginationRowModel,
    getSortedRowModel,
    flexRender,
} from '@tanstack/react-table'
import { rankItem } from '@tanstack/match-sorter-utils'
import { data100 } from '@/views/ui-components/data-display/Table/data'
import type { Person } from '@/views/ui-components/data-display/Table/data'
import type {
    ColumnDef,
    FilterFn,
    ColumnFiltersState,
} from '@tanstack/react-table'
import type { InputHTMLAttributes } from 'react'
import { Button } from '@/components/ui'
import { AiFillDelete, AiFillEdit, AiFillEye } from "react-icons/ai";
import { useNavigate } from 'react-router-dom'
import { FiLink } from 'react-icons/fi';
import { useAppSelector } from '@/store';
import BasicDialBox from '@/views/custom-compoents/confirmation-Dialog/BasicDialBox';
import useCustomToastify from '@/utils/customHook/useCustomToastify';


interface DebouncedInputProps
    extends Omit<
        InputHTMLAttributes<HTMLInputElement>,
        'onChange' | 'size' | 'prefix'
    > {
    value: string | number
    onChange: (value: string | number) => void
    debounce?: number

}

type Option = {
    value: number;
    label: string;
};
type AppDataType = {
    app_id: string,
    name: string,
    picture: string,
    app_url: string
}

const { Tr, Th, Td, THead, TBody, Sorter } = Table

function DebouncedInput({
    value: initialValue,
    onChange,
    debounce = 500,
    ...props
}: DebouncedInputProps) {
    const [value, setValue] = useState(initialValue)

    useEffect(() => {
        setValue(initialValue)
    }, [initialValue])

    useEffect(() => {
        const timeout = setTimeout(() => {
            onChange(value)
        }, debounce)

        return () => clearTimeout(timeout)
        // eslint-disable-next-line react-hooks/exhaustive-deps
    }, [value])

    return (

        <div className="flex items-center mb-4">
            <span className="mr-2">Search:</span>
            <Input
                {...props}
                value={value}
                onChange={(e) => setValue(e.target.value)}
            />
        </div>
    )
}

// eslint-disable-next-line @typescript-eslint/no-explicit-any
const fuzzyFilter: FilterFn<any> = (row, columnId, value, addMeta) => {
    // Rank the item
    const itemRank = rankItem(row.getValue(columnId), value)

    // Store the itemRank info
    addMeta({
        itemRank,
    })

    // Return if the item should be filtered in/out
    return itemRank.passed
}

const pageSizeOption = [
    { value: 10, label: '10 / page' },
    { value: 20, label: '20 / page' },
    { value: 30, label: '30 / page' },
    { value: 40, label: '40 / page' },
    { value: 50, label: '50 / page' },
];

const AppsTable = () => {
    const customToast = useCustomToastify()
    const theme = useAppSelector((state) => state.theme)
    const navigate = useNavigate();
    const [dialogBoxOpen, setDialogBoxOpen] = useState(false)
    const [columnFilters, setColumnFilters] = useState<ColumnFiltersState>([])
    // const [data] = useState(() => data100)
    const [data, setData] = useState<AppDataType[]>([])
    const [globalFilter, setGlobalFilter] = useState('')
    const [pageSize, setPageSize] = useState(10);
    const [signleAppDeatil, setSignleAppDeatil] = useState<AppDataType>()
    const [confirmationDialogBoxOpen, setConfirmationDialogBoxOpen] = useState<boolean>(false)
    const [deleteConfirmAppId, setDeleteConfirmAppId] = useState('')

    useEffect(() => {
        getAppList()
    }, [])

    const getAppList: any = async () => {
        try {
            const api = await fetch(`${import.meta.env.VITE_BASE_API_URL}/app/get-app-list`, {
                method: 'GET',
            })
            const response = await api.json()
            if (!response.error) {
                setData(response.data.result)
            }else{
                customToast.danger(response.message)
            }

        } catch (error) {
            console.error("Get App list API ERROR :", error)
        }
    }

    const columns = useMemo<ColumnDef<AppDataType>[]>(
        () => [
            {
                header: 'Sr No',
                accessorKey: 'sr-no',
                cell: ({ row }) => (
                    table.getState()
                        .pagination.pageSize * table
                            .getState()
                            .pagination
                        .pageIndex)
                    +
                    table.getRowModel().rows.indexOf(row) + 1,
            },
            {
                header: 'Image', accessorKey: 'image',
                cell: ({ row }) => <img src={`${row.original.picture}`} width={80} alt="image" />
            },
            {
                header: 'App Name', accessorKey: 'firstName',
                cell: ({ row }) => row.original.name
            },
            // { header: 'Last Name', accessorKey: 'lastName' },
            {
                header: 'App URL', accessorKey: 'email',
                cell: ({ row }) => row.original.app_url
            },
            {
                header: 'Action',
                accessorKey: 'action',
                cell: ({ row }) => (
                    <div className="flex gap-2"><Button size='xs' style={{ backgroundColor: '#e6e1e1', color: 'black' }} icon={<AiFillEye />} onClick={() => {
                        setSignleAppDeatil((prev) => ({ ...row.original }))
                        setDialogBoxOpen(true)
                    }} ></Button>
                        <Button
                            size='xs'
                            style={{ backgroundColor: '#0d8504', color: 'wheat' }}
                            icon={<AiFillEdit />}
                            onClick={() => {
                                navigate(`/auth/allApps/update/${row.original.app_id}`);
                            }} ></Button>
                        <Button
                            size='xs'
                            style={{ backgroundColor: '#bd0b0b', color: 'wheat' }}
                            icon={<AiFillDelete />}
                            onClick={() => {
                                setDeleteConfirmAppId(row.original.app_id)
                                setConfirmationDialogBoxOpen(true)
                            }}
                        ></Button>
                    </div>
                ),
            },
        ],
        []
    )



    const table = useReactTable({
        data,
        columns,
        filterFns: {
            fuzzy: fuzzyFilter,
        },
        state: {
            columnFilters,
            globalFilter,
        },
        onColumnFiltersChange: setColumnFilters,
        onGlobalFilterChange: setGlobalFilter,
        globalFilterFn: fuzzyFilter,
        getCoreRowModel: getCoreRowModel(),
        getFilteredRowModel: getFilteredRowModel(),
        getSortedRowModel: getSortedRowModel(),
        getPaginationRowModel: getPaginationRowModel(),
        getFacetedRowModel: getFacetedRowModel(),
        getFacetedUniqueValues: getFacetedUniqueValues(),
        getFacetedMinMaxValues: getFacetedMinMaxValues(),
        debugHeaders: true,
        debugColumns: false,
    })
    // console.log("Tabel : ", table.getRowModel().rows[0].getVisibleCells())
    const filteredData = table.getFilteredRowModel().rows.map(row => row.original);
    const totalFiltered = filteredData.length;

    useEffect(() => {
        if (table.getState().pagination.pageIndex >= Math.ceil(totalFiltered / pageSize)) {
            table.setPageIndex(Math.max(Math.ceil(totalFiltered / pageSize) - 1, 0));
        }
    }, [globalFilter, pageSize, totalFiltered, table]);

    const onPaginationChange = (page: number) => {
        table.setPageIndex(page - 1);
    };

    const onSelectChange = (value = 0) => {
        setPageSize(Number(value));
        table.setPageSize(Number(value));
    };

    console.log("current page index : ", table.getState().pagination.pageIndex + 1)
    console.log("Page Size : ", pageSize)

    const handleConfirm = async () => {
        try {

            const api = await fetch(`${import.meta.env.VITE_BASE_API_URL}/app/delete-app?app_id=${deleteConfirmAppId}`, {
                method: 'DELETE',
                //   body:formData
            })
            const response = await api.json()
            if (!response.error) {
                setConfirmationDialogBoxOpen(false)
                customToast.success("App Details Delete Successfully!")
                getAppList()
            } else {
                customToast.danger(response.message)
            }

        } catch (error) {
            console.error(error)
        }
    }

    const handleCancel = () => {
        setConfirmationDialogBoxOpen(false)
    }

    return (
        <>
            <div className={`${theme.mode == "dark" ? 'bg-[rgb(31,41,55)]' : 'bg-[rgb(255,255,255)]'} p-3 rounded-2xl`} >
                <div className='mt-3 mb-10 ml-4 flex items-center justify-between'>
                    <h3 className=''><b>All Apps</b></h3>
                    <Button className="mr-2 mb-2" size="sm" variant="solid" color={theme ? `${theme.themeColor}-${theme.primaryColorLevel}` : "blue-600"} onClick={() => { navigate('/auth/addApps') }}>
                        Add App
                    </Button>
                </div>


                <div className="flex justify-between">
                    <div style={totalFiltered > 0 ? { minWidth: 150 } : { minWidth: 150, visibility: "hidden" }} >
                        <Select<Option>
                            size="sm"
                            isSearchable={false}
                            value={pageSizeOption.find(
                                (option) =>
                                    option.value ===
                                    table.getState().pagination.pageSize,
                            )}
                            options={pageSizeOption}
                            onChange={(option) => onSelectChange(option?.value)}
                        />
                    </div>
                    <DebouncedInput
                        value={globalFilter ?? ''}
                        className="p-2 font-lg shadow border border-block w-72"
                        placeholder="Search all columns..."
                        onChange={(value) => setGlobalFilter(String(value))}
                    />
                </div>


                <Table>
                    <THead>
                        {table.getHeaderGroups().map((headerGroup) => (
                            <Tr key={headerGroup.id}>
                                {headerGroup.headers.map((header) => (
                                    <Th key={header.id} colSpan={header.colSpan} style={{ position: 'relative', width: header.getSize(), backgroundColor: `${theme.mode == 'light' ? '#f3f4f6' : 'rgb(55,65,81)'}` }}>
                                        {flexRender(header.column.columnDef.header, header.getContext())}
                                        {header.column.getCanResize() && (
                                            <div
                                                className={`table-resizer cursor-all-scroll ${header.column.getIsResizing() ? 'isResizing' : ''}`}
                                                onMouseDown={header.getResizeHandler()}
                                                onTouchStart={header.getResizeHandler()}
                                            ></div>
                                        )}
                                    </Th>

                                ))}
                                {/* <Th>
                                    Actions
                                </Th> */}
                            </Tr>
                        ))}
                    </THead>
                    <TBody>
                        {table.getRowModel().rows.map((row, index) => (
                            <Tr key={row.id}>
                                {row.getVisibleCells().map((cell) => (
                                    <Td key={cell.id} style={{ width: cell.column.getSize(), backgroundColor: `${index % 2 == 0 ? `${theme.mode == 'light' ? "white" : 'rgb(31,41,55)'}` : `${theme.mode == 'light' ? '#f3f4f6' : 'rgb(55,65,81)'}`}` }}>
                                        {cell.column.columnDef.cell === "image" ? (
                                            <img src={cell.getValue() as string} alt="image" />
                                        ) : (
                                            flexRender(cell.column.columnDef.cell, cell.getContext())
                                        )}
                                    </Td>
                                ))}
                            </Tr>
                        ))}
                    </TBody>
                </Table>
                {totalFiltered > 0 && (

                    <div className="flex items-center justify-between mt-4 mb-16">
                        <div>
                            Showing 1 to {table.getState().pagination.pageSize} of {filteredData.length} entries
                        </div>
                        <Pagination
                            pageSize={pageSize}
                            currentPage={table.getState().pagination.pageIndex + 1}
                            total={totalFiltered}
                            onChange={onPaginationChange}
                        />
                    </div>
                )}
            </div>
            <Dialog
                isOpen={dialogBoxOpen}
                bodyOpenClassName="overflow-hidden"
                onClose={() => {
                    setDialogBoxOpen(false)
                    setSignleAppDeatil(undefined)
                }}
                onRequestClose={() => {
                    setDialogBoxOpen(false)
                    setSignleAppDeatil(undefined)
                }}
                shouldCloseOnOverlayClick={false}
                width={650}
            >
                <h3 className="mb-8">App Inforamation</h3>
                <div >
                    <div className='flex gap-x-6 mb-3 items-center'>
                        <h4 className="text-lg font-semibold  w-40">App Name</h4>
                        <div className='text-lg'>{signleAppDeatil?.name}</div>
                    </div>
                    <div className='flex gap-x-6 mb-6'>
                        <h4 className='text-lg font-semibold w-40 '>App URL</h4>
                        <div className='text-base  hover:text-sky-500'><a target='_blank' className='flex gap-x-2 items-center' href={`${signleAppDeatil?.app_url}`}> <FiLink /> {signleAppDeatil?.app_url}</a></div>
                    </div>
                    <div className='flex gap-x-6'>
                        <h4 className='text-lg font-semibold w-40 '>App Image</h4>
                        <img src={`${import.meta.env.VITE_BASE_URL}${signleAppDeatil?.picture}`} alt="App Image" className='w-60' />
                    </div>
                </div>
                <div className="text-center mt-10">
                    <Button
                        className="ltr:mr-8 rtl:ml-8"
                        variant="twoTone"
                        color={theme ? `${theme.themeColor}-${theme.primaryColorLevel}` : "blue-600"}
                        onClick={() => {
                            setDialogBoxOpen(false)
                            navigate(`/auth/allApps/update/${signleAppDeatil?.app_id}`)
                        }}
                    >
                        Edit
                    </Button>
                    <Button variant="twoTone" color="red-600" onClick={() => {
                        setDeleteConfirmAppId(signleAppDeatil?.app_id || '')
                        setDialogBoxOpen(false)
                        setConfirmationDialogBoxOpen(true)
                    }}>
                        Delete
                    </Button>
                </div>
            </Dialog>

            <BasicDialBox
                dialogBoxOpen={confirmationDialogBoxOpen}
                onDialogClose={handleCancel}
                onDialogOk={handleConfirm}
                dialogHeader={"Confirmation of App Details Deleted"}
                dialogDescripation={"Do you really want to delete the app details?"}
            />
        </>
    )
}

export default AppsTable
