import { FormItem, FormContainer } from '@/components/ui/Form'
import Input from '@/components/ui/Input'
import Button from '@/components/ui/Button'
import Select from '@/components/ui/Select'
import Radio from '@/components/ui/Radio'
import Upload from '@/components/ui/Upload'
import * as Yup from 'yup'
import { Field, Form, Formik } from 'formik'
import type { FieldInputProps, FieldProps } from 'formik'
import { HiCheckCircle } from 'react-icons/hi'
import { useAppSelector } from '@/store'
import { useNavigate, useParams } from 'react-router-dom'
import useCustomToastify from '@/utils/customHook/useCustomToastify'
import { ComponentType, useEffect, useState } from 'react'
import { InputGroup } from '@/components/ui'
import { components, OptionProps, SingleValueProps } from 'react-select'
import { countryList } from '@/constants/countries.constant'
import { NumericFormat, NumericFormatProps } from 'react-number-format'
import type { InputProps } from '@/components/ui/Input'
import { FcImageFile } from 'react-icons/fc'
import { FetchData } from '@/utils/API Fech/Fetch Function/state_countyFetchFun'
import BasicDialBox from '@/views/custom-compoents/confirmation-Dialog/BasicDialBox'

const { SingleValue } = components

type CountryOption = {
    label: string
    dialCode: string
    value: string
}
type stateOptionAll = {
    value: number,
    label: string
}

const userRoles = import.meta.env.VITE_USER_ROLL.split(",").map((role: string, index: string) => {
    return {
        value: index,
        label: role.split("_").map((word: string) => word.charAt(0).toUpperCase() + word.slice(1)).join(" "),
    }
})

const statusOptions = [
    { value: 'active', label: 'Active' },
    { value: 'inactive', label: 'Inactive' },
]

const PhoneSelectOption = ({
    innerProps,
    data,
    isSelected,
}: OptionProps<CountryOption>) => {
    return (
        <div
            className={`cursor-pointer flex items-center justify-between p-2  w-60 ${isSelected
                ? 'bg-gray-100 dark:bg-gray-500'
                : 'hover:bg-gray-50 dark:hover:bg-gray-600'
                }`}
            {...innerProps}
        >
            <div className="flex items-center gap-2">
                <span>
                    ({data.label}) {data.dialCode}
                </span>
            </div>
        </div>
    )
}

const PhoneControl = (props: SingleValueProps<CountryOption>) => {
    const selected = props.getValue()[0]
    return (
        <SingleValue {...props}>
            {selected && <span>{selected.value} {selected.dialCode}</span>}
        </SingleValue>
    )
}

const NumericFormatInput = ({
    onValueChange,
    ...rest
}: Omit<NumericFormatProps, 'form'> & {
    /* eslint-disable @typescript-eslint/no-explicit-any */
    form: any
    field: FieldInputProps<unknown>
}) => {
    return (
        <NumericFormat
            customInput={Input as ComponentType}
            type="text"
            autoComplete="off"
            onValueChange={onValueChange}
            {...rest}
        />
    )
}

const NumberInput = (props: InputProps) => {
    return <Input {...props} value={props.field.value} />
}

const validationSchema = Yup.object().shape({
    picture: Yup.array().min(1, 'At least one file uploaded!'),
    status: Yup.string().required('Status is required'),
    firstName: Yup.string().required('First name is required'),
    lastName: Yup.string().required('Last name is required'),
    address: Yup.object().shape({
        street1: Yup.string().required('Street 1 is required'),
        street2: Yup.string(),
        city: Yup.string().required('City is required'),
        state: Yup.string().required('State is required'),
        country: Yup.string().required('Country is required'),
        zipCode: Yup.string().required('Zip code is required'),
    }),
    email: Yup.string().email('Invalid email address').required('Email address is required'),
    // mobile: Yup.string().required('Mobile number is required'),
    userRole: Yup.string().required('User role is required'),
    // sex: Yup.string().required('Sex is required'),
    dialCode: Yup.string().required('Please select dial code'),
    phoneNumber: Yup.string().required('Please enter your phone number'),
})

type SingleUserDeatilsType = {
    userId: string,
    firstName: string,
    lastName: string,
    picture: {
        imgUrl: string;
        imgName: string;
    }[];
    status: string,
    email: string,
    dialCode: string,
    phoneNumber: string,
    userRole: string,
    address: {
        street1: string,
        street2: string,
        city: string,
        state: string,
        country: string,
        zipCode: string
    }
}

// Define initial values
const initialValues = {
    userId: '',
    firstName: '',
    lastName: '',
    picture: [],
    status: '',
    email: '',
    dialCode: '',
    phoneNumber: '',
    userRole: '',
    address: {
        street1: '',
        street2: '',
        city: '',
        state: '',
        country: '',
        zipCode: '',
    },
}



const UpdateUser = () => {
    const { id } = useParams()
    const theme = useAppSelector((state) => state.theme)
    const navigate = useNavigate()
    const customToast = useCustomToastify()
    const [countryName, setCountryName] = useState<stateOptionAll[]>([])
    const [stateName, setStateName] = useState<stateOptionAll[]>([])
    const [cityName, setCityName] = useState<stateOptionAll[]>([])
    const [dialogBoxOpen, setDialogBoxOpen] = useState<boolean>(false)

    const [intialValueOfApp, setIntialValueOfApp] = useState<SingleUserDeatilsType>({ ...initialValues })

    useEffect(() => {
        getSingleUserData()
    }, [id])

    useEffect(() => {
        getCountryList()
    }, [])

    const getSingleUserData = async () => {
        try {
            const api = await fetch(`${import.meta.env.VITE_BASE_API_URL}/users/get-single-user?user_id=${id}`, {
                method: 'GET',
            })
            const response = await api.json()
            if (!response.error) {
                const tempData = {
                    userId: response.data.user_id,
                    firstName: response.data.full_name.split(" ")[0],
                    lastName: response.data.full_name.split(" ")[1],
                    picture: [
                        {
                            imgUrl: response.data.picture,
                            imgName: "Profile-Image"
                        }
                    ],
                    status: response.data.status == 1 ? 'active' : 'inactive',
                    email: response.data.email,
                    dialCode: response.data.dial_code,
                    phoneNumber: response.data.mobile,
                    userRole: response.data.user_role,
                    address: {
                        street1: response.data.street_1,
                        street2: response.data.street_2,
                        city: response.data.city,
                        state: response.data.state,
                        country: response.data.country,
                        zipCode: response.data.zip_code,
                    },
                }
                setIntialValueOfApp(tempData)
                getStateList(response.data.country)
                getCityList(response.data.state)

            } else {
                customToast.danger(response.message)
            }

        } catch (error) {
            console.error("Get App list API ERROR :", error)
        }
    }

    const getCountryList: any = async () => {
        const data = await FetchData("country_list")
        const tempData = data.map((item: any) => ({ value: item.id, label: item.name }))
        setCountryName((prev) => [...tempData])
    }

    const getStateList: any = async (countryId: number) => {
        const data = await FetchData(`state_list?country_id=${countryId}`)
        const tempData = data.map((item: any) => ({ value: item.id, label: item.name }))
        setStateName((prev) => [...tempData])
    }
    const getCityList: any = async (stateId: number) => {
        const data = await FetchData(`city_list?state_id=${stateId}`)
        const tempData = data.map((item: any) => ({ value: item.id, label: item.name }))
        setCityName((prev) => [...tempData])
    }

    const handleConfirm = async () => {
        // console.log('Form submitted:',intialValueOfApp )

        try {
            const formData = new FormData();
            //   formData.append("name", (intialValueOfApp?.appName || ''))
            //   formData.append("app_url", (intialValueOfApp?.appUrl || ''))
            formData.append('first_name', intialValueOfApp.firstName)
            formData.append('last_name', intialValueOfApp.lastName)
            formData.append('country', intialValueOfApp.address.country)
            formData.append('state', intialValueOfApp.address.state)
            formData.append('city', intialValueOfApp.address.city)
            formData.append('street_1', intialValueOfApp.address.street1)
            formData.append('street_2', intialValueOfApp.address.street2)
            formData.append('zip_code', intialValueOfApp.address.zipCode)
            formData.append('email', intialValueOfApp.email)
            formData.append('mobile', intialValueOfApp.phoneNumber)
            formData.append('user_role', intialValueOfApp.userRole)
            formData.append('dial_code', intialValueOfApp.dialCode)
            formData.append("status", (intialValueOfApp?.status === 'active' ? 1 : 0).toString())
            const file = intialValueOfApp?.picture.find((item) => item instanceof File);
            if (file) {
                formData.append('picture', file);
            }
            // formData.append("_method","PUT")
            for (const pair of formData.entries()) {
                console.log(pair[0], pair[1]);
            }
            // return
            const api = await fetch(`${import.meta.env.VITE_BASE_API_URL}/users/update-user?user_id=${intialValueOfApp?.userId}`, {
                method: 'PUT',
                body: formData
            })
            const response = await api.json()
            if (!response.error) {
                setDialogBoxOpen(false)
                customToast.success("User Details Update Successfully!")
                navigate('/auth/users/allUsers')
            } else {
                customToast.danger(response.message)
            }

        } catch (error) {
            console.error(error)
        }
    }


    const beforeUpload = (file: FileList | null, fileList: File[]) => {
        let valid: string | boolean = true

        const allowedFileType = ['image/jpeg', 'image/png']
        const MAX_FILE_SIZE = 500000

        if (fileList.length > 1) {
            return `You can only upload one file`
        }

        if (file) {
            for (const f of file) {
                if (!allowedFileType.includes(f.type)) {
                    valid = 'Please upload a .jpeg or .png file!'
                }

                if (f.size >= MAX_FILE_SIZE) {
                    valid = 'Upload image cannot be more than 500kb!'
                }
            }
        }

        return valid
    }

    const ShowIamgeDiv = () => {
        return <>
            <div className="upload-file">
                <div className="flex">
                    <div className="upload-file-thumbnail">
                        <img
                            className="upload-file-image"
                            src={`${intialValueOfApp?.picture[0]?.imgUrl}`}
                            alt={`file preview ${intialValueOfApp?.picture[0]?.imgName}`}
                        />
                    </div>
                    <div className="upload-file-info">
                        <h6 className="upload-file-name">{intialValueOfApp?.picture[0]?.imgName}</h6>
                        {/* <span className="upload-file-size">{intialValueOfApp.picture[0]?.imgSIze} kb</span> */}
                    </div>
                </div>
                <span onClick={() => {
                    setIntialValueOfApp((prev) => ({
                        ...prev,
                        ...initialValues
                    }));
                }} className="close-btn upload-file-remove" role="button"><svg stroke="currentColor" fill="currentColor" strokeWidth="0" viewBox="0 0 20 20" aria-hidden="true" height="1em" width="1em" xmlns="http://www.w3.org/2000/svg"><path fillRule="evenodd" d="M4.293 4.293a1 1 0 011.414 0L10 8.586l4.293-4.293a1 1 0 111.414 1.414L11.414 10l4.293 4.293a1 1 0 01-1.414 1.414L10 11.414l-4.293 4.293a1 1 0 01-1.414-1.414L8.586 10 4.293 5.707a1 1 0 010-1.414z" clipRule="evenodd"></path></svg></span>
            </div>
        </>
    }

    const handleCancel: () => void = () => {
        setDialogBoxOpen(false)  // Close confirmation dialog
    }

    return (
        <>
            <div className={` p-3 rounded-2xl`} >
                <h2 className='mt-3 mb-5 ml-3'>Add User Details</h2>

                <div className='px-3'>
                    <Formik
                        initialValues={intialValueOfApp}
                        validationSchema={validationSchema}
                        enableReinitialize={true} // Add this prop for when state is change
                        onSubmit={(values, { resetForm, setSubmitting }) => {
                            try {
                                console.log("jen", values)
                                setIntialValueOfApp((prev) => ({ ...values }))
                                setDialogBoxOpen(true)
                                setSubmitting(false)
                                resetForm()
                                // storeUserDeatils(values, setSubmitting, resetForm)

                            } catch (error) {
                                console.log("Error: ", error)
                            }
                        }}
                    >
                        {({ values, touched, errors, resetForm }) => (
                            <Form>
                                {/* style={{ gridTemplateColumns: '60% 28%' }} */}
                                <FormContainer>
                                    <div className="grid lg:grid-cols-2 md:grid-cols-2 grid-cols-1 gap-7 "  >
                                        <div className={`border rounded-md ${theme.mode == "dark" ? 'bg-[rgb(31,41,55)]' : 'bg-[rgb(255,255,255)]'}`}>
                                            <div className={`text-2xl mb-5 h-fit rounded-t-md bg-[rgb(227,231,236)] p-2 ${theme.mode == "dark" ? "text-black" : null}`}>Basic Deatils</div>
                                            <div className='p-2 px-4'>
                                                <FormItem asterisk
                                                    label="First Name"
                                                    invalid={errors.firstName && touched.firstName}
                                                    errorMessage={errors.firstName}>
                                                    <Field
                                                        type="text"
                                                        name="firstName"
                                                        placeholder="First Name"
                                                        component={Input}
                                                        size='md' />
                                                </FormItem>

                                                <FormItem asterisk label="Last Name" invalid={errors.lastName && touched.lastName} errorMessage={errors.lastName}>
                                                    <Field type="text" name="lastName" placeholder="Last Name" component={Input} size='md' />
                                                </FormItem>
                                                <div className="grid lg:grid-cols-2 md:grid-cols-2 grid-cols-1 gap-4">
                                                    <FormItem asterisk label="User Role" invalid={errors.userRole && touched.userRole} errorMessage={errors.userRole}>
                                                        <Field name="userRole" >
                                                            {({ field, form }: FieldProps<any>) => (
                                                                <Select
                                                                    field={field}
                                                                    form={form}
                                                                    size='md'
                                                                    options={userRoles}
                                                                    value={userRoles.find((option: any) => option.value == values.userRole)}
                                                                    onChange={(option) => form.setFieldValue(field.name, option?.value)}
                                                                />
                                                            )}
                                                        </Field>
                                                    </FormItem>
                                                </div>
                                            </div>
                                        </div>
                                        <div>
                                            <div className={`border rounded-md h-full ${theme.mode == "dark" ? 'bg-[rgb(31,41,55)]' : 'bg-[rgb(255,255,255)]'}`}>
                                                <div className={`text-2xl mb-5 h-fit rounded-t-md bg-[rgb(227,231,236)] p-2 ${theme.mode == "dark" ? "text-black" : null}`}>Profile Image</div>
                                                <div className='p-2 px-4'>
                                                    <FormItem asterisk label="Picture" invalid={Boolean(errors.picture && touched.picture)} errorMessage={errors.picture as string}>
                                                        <Field name="picture" >
                                                            {({ field, form }: FieldProps<any>) => (
                                                                <Upload draggable beforeUpload={beforeUpload}
                                                                    onChange={(files) => {
                                                                        setIntialValueOfApp((prev) => ({ ...prev, picture: [] }))
                                                                        form.setFieldValue(field.name, files)
                                                                    }}
                                                                    onFileRemove={(files) => form.setFieldValue(field.name, files)} uploadLimit={1} >
                                                                    <div className="my-7 text-center">
                                                                        <div className="text-6xl mb-4 flex justify-center">
                                                                            <FcImageFile />
                                                                        </div>
                                                                        <p className="font-semibold">
                                                                            <span className="text-gray-800 dark:text-white">
                                                                                Drop your image here, or{' '}
                                                                            </span>
                                                                            <span className="text-blue-500">browse</span>
                                                                        </p>
                                                                        <p className="mt-1 opacity-60 dark:text-white">
                                                                            Support: jpeg, png, gif
                                                                        </p>
                                                                    </div>
                                                                </Upload>
                                                            )}
                                                        </Field>
                                                        {
                                                            intialValueOfApp && intialValueOfApp?.picture.length > 0 ? <>
                                                                <ShowIamgeDiv />
                                                            </>
                                                                :
                                                                null
                                                        }
                                                    </FormItem>
                                                </div>
                                            </div>

                                        </div>
                                    </div>
                                    <div className="grid lg:grid-cols-2 md:grid-cols-2 grid-cols-1 gap-7 mt-7 "  >
                                        <div>
                                            <div className={`border rounded-md ${theme.mode == "dark" ? 'bg-[rgb(31,41,55)]' : 'bg-[rgb(255,255,255)]'}`}>
                                                <div className={`text-2xl mb-5 h-fit bg-[rgb(227,231,236)] rounded-t-md p-2 ${theme.mode == "dark" ? "text-black" : null}`}>User Address</div>
                                                <FormItem className='p-2 px-4' label="" >
                                                    <Field name="address.street1">
                                                        {({ field }: FieldProps<any>) => (
                                                            <FormItem className='mb-4' asterisk label="Street 1" invalid={errors.address?.street1 && touched.address?.street1} errorMessage={errors.address?.street1}>
                                                                <Input {...field} placeholder="Street 1" size='md' />
                                                            </FormItem>
                                                        )}
                                                    </Field>

                                                    <Field name="address.street2">
                                                        {({ field }: FieldProps<any>) => (
                                                            <FormItem className='mb-4' label="Street 2" invalid={errors.address?.street2 && touched.address?.street2} errorMessage={errors.address?.street2}>
                                                                <Input {...field} placeholder="Street 2" size='md' />
                                                            </FormItem>
                                                        )}
                                                    </Field>

                                                    <div className="grid lg:grid-cols-2 md:grid-cols-2 grid-cols-1 gap-4">

                                                        <Field name="address.country">
                                                            {({ field, form }: FieldProps<any>) => (
                                                                <FormItem asterisk label="Country" invalid={errors.address?.country && touched.address?.country} errorMessage={errors.address?.country}>

                                                                    <Select
                                                                        field={field}
                                                                        form={form}
                                                                        size='md'
                                                                        options={countryName}
                                                                        value={countryName.find((option: any) => option.value === values.address.country)}
                                                                        onChange={(option) => {
                                                                            if (option) {
                                                                                alert("hello")
                                                                                form.setFieldValue(field.name, option?.value)
                                                                                form.setFieldValue('address.state', '');
                                                                                form.setFieldValue('address.city', '');
                                                                                getStateList(option?.value)
                                                                            } else {
                                                                                customToast.warning("Please select a country")
                                                                            }
                                                                        }}
                                                                    />

                                                                </FormItem>
                                                            )}
                                                        </Field>

                                                        <Field name="address.state">
                                                            {({ field, form }: FieldProps<any>) => (
                                                                <FormItem asterisk label="State" invalid={errors.address?.state && touched.address?.state} errorMessage={errors.address?.state}>

                                                                    <Select
                                                                        field={field}
                                                                        form={form}
                                                                        size='md'
                                                                        options={stateName}
                                                                        value={stateName.find((option: any) => option.value === values.address.state)}
                                                                        onChange={(option) => {
                                                                            if (touched.address?.country && option) {
                                                                                customToast.warning("Please select a country")
                                                                            } else if (option) {
                                                                                form.setFieldValue(field.name, option?.value)
                                                                                form.setFieldValue('address.city', '');
                                                                                setCityName([])
                                                                                getCityList(option?.value)
                                                                            }
                                                                        }}
                                                                    />

                                                                </FormItem>
                                                            )}
                                                        </Field>

                                                        <Field name="address.city">
                                                            {({ field, form }: FieldProps<any>) => (
                                                                <FormItem asterisk label="City" invalid={errors.address?.city && touched.address?.city} errorMessage={errors.address?.city}>

                                                                    <Select
                                                                        field={field}
                                                                        form={form}
                                                                        size='md'
                                                                        options={cityName}
                                                                        value={cityName.find((option: any) => option.value === values.address.city)}
                                                                        onChange={(option) => {
                                                                            form.setFieldValue(field.name, option?.value)
                                                                        }}
                                                                    />

                                                                </FormItem>
                                                            )}
                                                        </Field>

                                                        <Field name="address.zipCode">
                                                            {({ field }: FieldProps<any>) => (
                                                                <FormItem asterisk label="Zip Code" invalid={errors.address?.zipCode && touched.address?.zipCode} errorMessage={errors.address?.zipCode}>
                                                                    <Input {...field} placeholder="Zip Code" size='md' />
                                                                </FormItem>
                                                            )}
                                                        </Field>
                                                    </div>
                                                </FormItem>
                                            </div>
                                        </div>
                                        <div>
                                            <div className={`border  rounded-md ${theme.mode == "dark" ? 'bg-[rgb(31,41,55)]' : 'bg-[rgb(255,255,255)]'}`}>
                                                <div className={`text-2xl mb-5 h-fit bg-[rgb(227,231,236)] rounded-t-md p-2 ${theme.mode == "dark" ? "text-black" : null}`}>Contact Deatils</div>
                                                <div className='p-2 px-4'>
                                                    <FormItem asterisk label="Email Address" invalid={errors.email && touched.email} errorMessage={errors.email}>
                                                        <Field type="email" name="email" placeholder="Email Address" component={Input} size='md' />
                                                    </FormItem>
                                                    <FormItem
                                                        asterisk
                                                        label="Phone Number"
                                                        invalid={
                                                            (errors.dialCode &&
                                                                touched.dialCode) ||
                                                            (errors.phoneNumber &&
                                                                touched.phoneNumber)
                                                        }
                                                        errorMessage="Please enter your phone number"
                                                    >
                                                        <InputGroup>
                                                            <Field name="dialCode">

                                                                {({
                                                                    field,
                                                                    form,
                                                                }: FieldProps) => (
                                                                    <Select<CountryOption>
                                                                        className="min-w-[150px]"
                                                                        placeholder="Dial Code"
                                                                        components={{
                                                                            Option: PhoneSelectOption,
                                                                            SingleValue:
                                                                                PhoneControl,
                                                                        }}
                                                                        field={field}
                                                                        form={form}
                                                                        options={countryList}
                                                                        value={countryList.filter(
                                                                            (country) =>
                                                                                country.dialCode ===
                                                                                values.dialCode
                                                                        )}
                                                                        onChange={(country) =>
                                                                            form.setFieldValue(
                                                                                field.name,
                                                                                country?.dialCode
                                                                            )
                                                                        }
                                                                    />
                                                                )}

                                                            </Field>
                                                            <Field name="phoneNumber">
                                                                {({
                                                                    field,
                                                                    form,
                                                                }: FieldProps) => {
                                                                    return (
                                                                        <NumericFormatInput
                                                                            form={form}
                                                                            field={field}
                                                                            customInput={
                                                                                NumberInput as ComponentType
                                                                            }
                                                                            placeholder="Phone Number"
                                                                            onValueChange={(
                                                                                e
                                                                            ) => {
                                                                                form.setFieldValue(
                                                                                    field.name,
                                                                                    e.value
                                                                                )
                                                                            }}
                                                                        />
                                                                    )
                                                                }}
                                                            </Field>
                                                        </InputGroup>
                                                    </FormItem>
                                                </div>
                                            </div>
                                            <div className={`border mt-7 rounded-md ${theme.mode == "dark" ? 'bg-[rgb(31,41,55)]' : 'bg-[rgb(255,255,255)]'}`}>
                                                <div className={`text-2xl rounded-t-md mb-5 h-fit bg-[rgb(227,231,236)] p-2 ${theme.mode == "dark" ? "text-black" : null}`}>Status</div>
                                                <FormItem className='p-2 px-4' asterisk label="Status" invalid={errors.status && touched.status} errorMessage={errors.status}>
                                                    <Field name="status">
                                                        {({ field, form }: FieldProps<any>) => (
                                                            <Radio.Group value={values.status} onChange={(val) => form.setFieldValue(field.name, val)}>
                                                                {statusOptions.map(status => (
                                                                    <Radio key={status.value} value={status.value}>{status.label}</Radio>
                                                                ))}
                                                            </Radio.Group>
                                                        )}
                                                    </Field>
                                                </FormItem>
                                            </div>
                                        </div>
                                    </div>


                                    <FormItem className='text-center mt-7'>
                                        <Button
                                            type="reset"
                                            className="ltr:mr-7 rtl:ml-7"
                                            onClick={() => {
                                                resetForm()
                                                navigate('/auth/users/allUsers')
                                            }}
                                        >
                                            Reset
                                        </Button>
                                        <Button variant="solid" type="submit">
                                            Submit
                                        </Button>
                                    </FormItem>

                                </FormContainer>
                            </Form>
                        )}
                    </Formik>
                </div>
            </div>

            <BasicDialBox
                dialogBoxOpen={dialogBoxOpen}
                onDialogClose={handleCancel}
                onDialogOk={handleConfirm}
                dialogHeader={"Confirmation of Updated App Details"}
                dialogDescripation={"Do you really want to update the app's details?"}
            />
        </>
    )
}

export default UpdateUser