import { useState } from 'react'
import { FormItem, FormContainer } from '@/components/ui/Form'
import Input from '@/components/ui/Input'
import Button from '@/components/ui/Button'
import Upload from '@/components/ui/Upload'
import Checkbox from '@/components/ui/Checkbox'
import Radio from '@/components/ui/Radio'
import { Field, Form, Formik } from 'formik'
import { HiOutlineEyeOff, HiOutlineEye } from 'react-icons/hi'
import * as Yup from 'yup'
import type { FieldProps } from 'formik'
import type { MouseEvent } from 'react'
import { useNavigate } from 'react-router-dom'
import Notification from '@/components/ui/Notification'
import toast from '@/components/ui/toast'
import useCustomToastify from '@/utils/customHook/useCustomToastify'
import { useAppSelector } from '@/store'

const validationSchema = Yup.object().shape({
    // email: Yup.string().email('Invalid email').required('Email Required'),

    appName: Yup.string()
        .min(2, 'Too Short!')
        .max(12, 'Too Long!')
        .required('App Name Required'),
    picture: Yup.array().required('At least one picture is required')
        .required("App Image is required"),

    appUrl: Yup.string().url('Invalid URL')
        .required('App URL Required'),
    status: Yup.string().required('Status is required'),
    descripation:Yup.string().required(),
})
const statusOptions = [
    { value: 'active', label: 'Active' },
    { value: 'inactive', label: 'Inactive' },
]

interface valuesParameter {
    appName: string,
    appUrl: string,
    picture: any,
    status: string,
    descripation:string,
}
const AppVallidationForm = () => {
    const theme = useAppSelector((state) => state.theme)
    const navigate = useNavigate()
    const customToast = useCustomToastify()
    const [pwInputType, setPwInputType] = useState('password')

    const postAppDeatilsAPI: any = async ({ appName, appUrl, picture, status,descripation }: valuesParameter,setSubmitting:any,resetForm:any) => {
        try {
            console.table([appName, appUrl, picture, status])
            console.log("FILE",picture[0])
            const formData = new FormData()
            formData.append('name', appName)
            formData.append('app_url', appUrl)
            formData.append('picture', picture[0])
            formData.append('status', (status === 'active' ? 1 : 0).toString())
            formData.append('app_description',descripation)
            // return
            const api = await fetch(`${import.meta.env.VITE_BASE_API_URL}/app/add-app`, {
                method: 'POST',
                body: formData
            })
            const response = await api.json()
            if (!response.error) {
                customToast.success("App Deatils Added Successfully!")
                setSubmitting(false)
                resetForm()
                navigate('/auth/allApps')
            } else {
                customToast.danger(response.message)
            }
        } catch (error) {
            console.error(error)
        }
    }

    const onPasswordVisibleClick = (e: MouseEvent) => {
        e.preventDefault()
        setPwInputType(pwInputType === 'password' ? 'text' : 'password')
    }

    const passwordVisible = (
        <span
            className="cursor-pointer"
            onClick={(e) => onPasswordVisibleClick(e)}
        >
            {pwInputType === 'password' ? (
                <HiOutlineEyeOff />
            ) : (
                <HiOutlineEye />
            )}
        </span>
    )
    const beforeUpload = (file: FileList | null, fileList: File[]) => {
        let valid: string | boolean = true

        const allowedFileType = ['image/jpeg', 'image/png']
        const MAX_FILE_SIZE = 500000

        if (fileList.length > 1) {
            return `You can only upload one file`
        }

        if (file) {
            for (const f of file) {
                if (!allowedFileType.includes(f.type)) {
                    valid = 'Please upload a .jpeg or .png file!'
                }

                if (f.size >= MAX_FILE_SIZE) {
                    valid = 'Upload image cannot be more than 500kb!'
                }
            }
        }

        return valid
    }
    return (
        <>
            <div className={`${theme.mode == "dark" ? 'bg-[rgb(31,41,55)]' : 'bg-[rgb(255,255,255)]'} p-3 rounded-2xl`} >
                <h3 className='mt-3 mb-5 ml-4'><b>Add Apps</b></h3>

                <div>
                    <Formik
                        initialValues={{
                            appName: '',
                            picture: [],
                            appUrl: '',
                            status: '',
                            descripation:'',
                        }}
                        validationSchema={validationSchema}
                        onSubmit={(values, { resetForm, setSubmitting }) => {
                            try {
                                // console.log("jen", values)
                                // setSubmitting(false)
                                // resetForm()
                                postAppDeatilsAPI(values,setSubmitting,resetForm)

                            } catch (error) {
                                console.log("Error: ", error)
                            }
                        }}
                    >
                        {({ values, touched, errors, resetForm }) => (
                            <Form>
                                <FormContainer>
                                    <FormItem
                                        asterisk
                                        label="App Name"
                                        invalid={errors.appName && touched.appName}
                                        errorMessage={errors.appName}
                                    >
                                        <Field
                                            type="text"
                                            autoComplete="off"
                                            name="appName"
                                            placeholder="App Name"
                                            component={Input}
                                        />
                                    </FormItem>
                                     <FormItem
                                        asterisk
                                        label="App description"
                                        invalid={errors.descripation && touched.descripation}
                                        errorMessage={errors.descripation}
                                    >
                                        <Field
                                            type="text"
                                            autoComplete="off"
                                            name="descripation"
                                            placeholder="App Description"
                                            component={Input}
                                        />
                                    </FormItem>
                                    
                                    <FormItem
                                        asterisk
                                        label="App URL"
                                        invalid={errors.appUrl && touched.appUrl}
                                        errorMessage={errors.appUrl}
                                    >
                                        <Field
                                            type="text"
                                            autoComplete="off"
                                            name="appUrl"
                                            placeholder="App URL"
                                            component={Input}
                                        />
                                    </FormItem>
                                 
                                    <FormItem asterisk label="Picture" invalid={Boolean(errors.picture && touched.picture)} errorMessage={errors.picture as string}>
                                        <Field name="picture" >
                                            {({ field, form }: FieldProps<any>) => (
                                                <Upload beforeUpload={beforeUpload} fileList={values.picture}
                                                    onChange={(files) => form.setFieldValue(field.name, files)}
                                                    onFileRemove={(files) => form.setFieldValue(field.name, files)} uploadLimit={1} />
                                            )}
                                        </Field>
                                    </FormItem>

                                    <FormItem asterisk label="Status" invalid={errors.status && touched.status} errorMessage={errors.status}>
                                        <Field name="status">
                                            {({ field, form }: FieldProps<any>) => (
                                                <Radio.Group value={values.status} onChange={(val) => form.setFieldValue(field.name, val)}>
                                                    {statusOptions.map(status => (
                                                        <Radio key={status.value} value={status.value}>{status.label}</Radio>
                                                    ))}
                                                </Radio.Group>
                                            )}
                                        </Field>
                                    </FormItem>

                                    <FormItem>
                                        <Button
                                            type="reset"
                                            className="ltr:mr-2 rtl:ml-2"
                                            onClick={() => {
                                                resetForm()
                                                navigate('/allApps')
                                            }}
                                        >
                                            Reset
                                        </Button>
                                        <Button variant="solid" type="submit">
                                            Submit
                                        </Button>
                                    </FormItem>

                                </FormContainer>
                            </Form>
                        )}
                    </Formik>
                </div>
            </div>

        </>
    )
}

export default AppVallidationForm