import { useEffect, useState } from 'react'
import { FormItem, FormContainer } from '@/components/ui/Form'
import Input from '@/components/ui/Input'
import Button from '@/components/ui/Button'
import Upload from '@/components/ui/Upload'
import Checkbox from '@/components/ui/Checkbox'
import Radio from '@/components/ui/Radio'
import { Field, Form, Formik } from 'formik'
import { HiOutlineEyeOff, HiOutlineEye } from 'react-icons/hi'
import * as Yup from 'yup'
import type { FieldProps } from 'formik'
import type { MouseEvent } from 'react'
import { useNavigate, useParams } from 'react-router-dom'
import BasicDialBox from '@/views/custom-compoents/confirmation-Dialog/BasicDialBox'
import useCustomToastify from '@/utils/customHook/useCustomToastify'
import { useAppSelector } from '@/store'

// Validation Schema
const validationSchema = Yup.object().shape({
  appName: Yup.string()
    .min(2, 'Too Short!')
    .max(20, 'Too Long!')
    .required('App Name Required'),
  picture: Yup.array().required('At least one picture is required')
    .required("App Image is required"),

  appUrl: Yup.string().url('Invalid URL')
    .required('App URL Required'),
  status: Yup.string().required('Status is required'),
  app_description:Yup.string().required("App description is a required field")
})
const statusOptions = [
  { value: 'active', label: 'Active' },
  { value: 'inactive', label: 'Inactive' },
]

type SingleAppDeatilsType = {
  appId: string,
  appName: string;
  picture: {
    imgUrl: string;
    imgName: string;
  }[];
  appUrl: string;
  status: string;
  app_description:string
}

// Comonent Start 
const UpdateApps = () => {
  const theme = useAppSelector((state) => state.theme)
  const customToast = useCustomToastify()
  const navigate = useNavigate()
  const { id } = useParams()
  const [intialValueOfApp, setIntialValueOfApp] = useState<SingleAppDeatilsType>()

  const [dialogBoxOpen, setDialogBoxOpen] = useState<boolean>(false)
  const [formValueData, setFormValueData] = useState({})

  useEffect(() => {
    getSingleAppData()
  }, [id])

  const getSingleAppData = async () => {
    try {
      const api = await fetch(`${import.meta.env.VITE_BASE_API_URL}/app/get-single-app?app_id=${id}`, {
        method: 'GET',
      })
      const response = await api.json()
      if (!response.error) {
        const tempData = {
          appName: response.data.name,
          appId: response.data.app_id,
          picture: [
            {
              imgUrl: response.data.picture,
              imgName: response.data.name,
            }
          ],
          appUrl: response.data.app_url,
          status: response.data.status == 1 ? 'active' : 'inactive',
          app_description:response.data.app_description
        }
        setIntialValueOfApp(tempData)
      } else {
        customToast.danger(response.message)
      }

    } catch (error) {
      console.error("Get App list API ERROR :", error)
    }
  }

  // File handling locally 
  const beforeUpload = (file: FileList | null, fileList: File[]) => {
    let valid: string | boolean = true

    const allowedFileType = ['image/jpeg', 'image/png']
    const MAX_FILE_SIZE = 500000

    if (fileList.length > 1) {
      return `You can only upload one file`
    }

    if (file) {
      for (const f of file) {
        if (!allowedFileType.includes(f.type)) {
          valid = 'Please upload a .jpeg or .png file!'
        }

        if (f.size >= MAX_FILE_SIZE) {
          valid = 'Upload image cannot be more than 500kb!'
        }
      }
    }

    return valid
  }

  const ShowIamgeDiv = () => {
    return <>
      <div className="upload-file">
        <div className="flex">
          <div className="upload-file-thumbnail">
            <img
              className="upload-file-image"
              src={`${intialValueOfApp?.picture[0]?.imgUrl}`}
              alt={`file preview ${intialValueOfApp?.picture[0]?.imgName}`}
            />
          </div>
          <div className="upload-file-info">
            <h6 className="upload-file-name">{intialValueOfApp?.picture[0]?.imgName}</h6>
            {/* <span className="upload-file-size">{intialValueOfApp.picture[0]?.imgSIze} kb</span> */}
          </div>
        </div>
        <span onClick={() => {
          setIntialValueOfApp((prev) => ({
            ...prev,
            picture: [],
            appName: prev ? prev.appName : '',
            appUrl: prev ? prev.appUrl : '',
            status: prev ? prev.status : '',
            appId: prev ? prev.appId : '',
            app_description:prev? prev.app_description :''
          }));
        }} className="close-btn upload-file-remove" role="button"><svg stroke="currentColor" fill="currentColor" strokeWidth="0" viewBox="0 0 20 20" aria-hidden="true" height="1em" width="1em" xmlns="http://www.w3.org/2000/svg"><path fillRule="evenodd" d="M4.293 4.293a1 1 0 011.414 0L10 8.586l4.293-4.293a1 1 0 111.414 1.414L11.414 10l4.293 4.293a1 1 0 01-1.414 1.414L10 11.414l-4.293 4.293a1 1 0 01-1.414-1.414L8.586 10 4.293 5.707a1 1 0 010-1.414z" clipRule="evenodd"></path></svg></span>
      </div>
    </>
  }

  const handleConfirm = async () => {
    // console.log('Form submitted:',intialValueOfApp )

    try {
      const formData = new FormData();
      formData.append("name", (intialValueOfApp?.appName || ''))
      formData.append("app_url", (intialValueOfApp?.appUrl || ''))
      formData.append("status", (intialValueOfApp?.status === 'active' ? 1 : 0).toString())
      formData.append('app_description',(intialValueOfApp?.app_description || ''))

      const file = intialValueOfApp?.picture.find((item) => item instanceof File);
      if (file) {
        formData.append('picture', file);
      }
      // formData.append("_method","PUT")
      for (const pair of formData.entries()) {
        console.log(pair[0], pair[1]);
      }
      // return
      const api = await fetch(`${import.meta.env.VITE_BASE_API_URL}/app/update-app?app_id=${intialValueOfApp?.appId}`, {
        method: 'PUT',
        body:formData
      })
      const response = await api.json()
      if (!response.error) {
        setDialogBoxOpen(false)
        customToast.success("App Details Update Successfully!")
        navigate('/auth/allApps')
      } else {
        customToast.danger(response.message)
      }

    } catch (error) {
      console.error(error)
    }
  }

  const handleCancel: () => void = () => {
    setDialogBoxOpen(false)  // Close confirmation dialog
  }

  return (
    <>
      <div className={`${theme.mode == "dark" ? 'bg-[rgb(31,41,55)]' : 'bg-[rgb(255,255,255)]'} p-3 rounded-2xl`} >
        <h3 className='mt-3 mb-5 ml-4'><b>Update Apps Details</b></h3>

        <div>
          <Formik
            initialValues={intialValueOfApp as SingleAppDeatilsType}
            validationSchema={validationSchema}
            enableReinitialize={true} // Add this prop for when state is change
            onSubmit={(values, { resetForm, setSubmitting }) => {


              // setFormValueData(values)

              try {
                if (values.picture.length == 0) {
                  customToast.warning("APP Image is required")
                  return
                }
                console.log("jen", values)
                setIntialValueOfApp((prev) => ({ ...values }))
                setDialogBoxOpen(true)
                setSubmitting(false)
                resetForm()
                // setTimeout(() => {
                //     alert(JSON.stringify(values, null, 2))
                //     
                // }, 400)
              } catch (error) {
                console.log("Error: ", error)
              }
            }}
          >
            {({ values, touched, errors, resetForm }) => (
              <Form>
                <FormContainer>
                  <FormItem
                    asterisk
                    label="App Name"
                    invalid={errors.appName && touched.appName}
                    errorMessage={errors.appName}
                  >
                    <Field
                      type="text"
                      autoComplete="off"
                      name="appName"
                      placeholder="App Name"
                      component={Input}
                    />
                  </FormItem>
                  <FormItem
                    asterisk
                    label="App description"
                    invalid={errors.app_description && touched.app_description}
                    errorMessage={errors.app_description}
                  >
                    <Field
                      type="text"
                      autoComplete="off"
                      name="app_description"
                      placeholder="App descripation"
                      component={Input}
                    />
                  </FormItem>
                  <FormItem
                    asterisk
                    label="App URL"
                    invalid={errors.appUrl && touched.appUrl}
                    errorMessage={errors.appUrl}
                  >
                    <Field
                      type="text"
                      autoComplete="off"
                      name="appUrl"
                      placeholder="App URL"
                      component={Input}
                    />
                  </FormItem>
                  {/* Image Upload Code */}
                  <FormItem asterisk label="Picture" invalid={Boolean(errors.picture && touched.picture)} errorMessage={errors.picture as string}>
                    <Field name="picture" >
                      {({ field, form }: FieldProps<any>) => (
                        <Upload beforeUpload={beforeUpload}
                          onChange={(files) => {
                            setIntialValueOfApp((prev) => ({
                              ...prev,
                              picture: [],
                              appName: prev ? prev.appName : '',
                              appUrl: prev ? prev.appUrl : '',
                              status: prev ? prev.status : '',
                              appId: prev ? prev.appId : '',
                              app_description:prev? prev.app_description :''
                            }));
                            form.setFieldValue(field.name, files)
                          }}
                          onFileRemove={(files) => form.setFieldValue(field.name, files)} uploadLimit={1} />
                      )}
                    </Field>
                    {
                      intialValueOfApp && intialValueOfApp?.picture.length > 0 ? <>
                        <ShowIamgeDiv />
                      </>
                        :
                        null
                    }

                  </FormItem>
                  {/* Status Chaneg Code */}
                  <FormItem asterisk label="Status" invalid={errors.status && touched.status} errorMessage={errors.status}>
                    <Field name="status">
                      {({ field, form }: FieldProps<any>) => (
                        <Radio.Group value={values?.status} onChange={(val) => form.setFieldValue(field.name, val)}>
                          {statusOptions.map(status => (
                            <Radio key={status.value} value={status.value}>{status.label}</Radio>
                          ))}
                        </Radio.Group>
                      )}
                    </Field>
                  </FormItem>
                  <FormItem>
                    <Button
                      type="reset"
                      className="ltr:mr-2 rtl:ml-2"
                      onClick={() => {
                        resetForm()
                        navigate('/auth/allApps')
                      }}
                    >
                      Reset
                    </Button>
                    <Button variant="solid" type="submit">
                      Submit
                    </Button>
                  </FormItem>
                </FormContainer>
              </Form>
            )}
          </Formik>
        </div>
      </div>

      <BasicDialBox
        dialogBoxOpen={dialogBoxOpen}
        onDialogClose={handleCancel}
        onDialogOk={handleConfirm}
        dialogHeader={"Confirmation of Updated App Details"}
        dialogDescripation={"Do you really want to update the app's details?"}
      />
    </>
  )
}

export default UpdateApps

